<?php
require_once 'config.php';

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    echo "<div class='text-center py-4 text-red-500'>Access denied. Please login as admin.</div>";
    exit;
}

// Use secure connection
$conn = getSecureConnection();

// Check if order ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    echo "<div class='text-center py-4 text-red-500'>Order ID not provided.</div>";
    exit;
}

$order_id = intval($_GET['id']);

// Get order from database
$sql = "SELECT * FROM orders WHERE id = ?";
$stmt = $conn->prepare($sql);
if (!$stmt) {
    echo "<div class='text-center py-4 text-red-500'>Database error: " . $conn->error . "</div>";
    exit;
}

$stmt->bind_param("i", $order_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows == 0) {
    echo "<div class='text-center py-4 text-red-500'>Order not found.</div>";
    exit;
}

$order = $result->fetch_assoc();
$stmt->close();
$conn->close();
?>

<div class="space-y-6">
    <!-- Order Header -->
    <div class="border-b border-gray-200 pb-4">
        <div class="flex justify-between items-start">
            <div>
                <h3 class="text-lg font-medium text-gray-900">
                    <i class="fas fa-hashtag mr-2"></i>Order #<?php echo htmlspecialchars($order['order_id']); ?>
                </h3>
                <p class="text-sm text-gray-500 mt-1">
                    <i class="fas fa-calendar mr-1"></i>Placed on <?php echo date('M d, Y \a\t H:i', strtotime($order['order_date'])); ?>
                </p>
            </div>
            <span class="px-3 py-1 text-sm font-medium rounded-full <?php
                $statusClass = [
                    'pending' => 'bg-yellow-100 text-yellow-800',
                    'processing' => 'bg-blue-100 text-blue-800',
                    'completed' => 'bg-green-100 text-green-800',
                    'cancelled' => 'bg-red-100 text-red-800'
                ][$order['status']] ?? 'bg-gray-100 text-gray-800';
                echo $statusClass;
            ?>">
                <?php echo ucfirst(htmlspecialchars($order['status'])); ?>
            </span>
        </div>
    </div>

    <!-- Order Details Grid -->
    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
        <!-- Customer Information -->
        <div class="bg-gray-50 p-4 rounded-lg">
            <h4 class="text-md font-medium text-gray-900 mb-3">
                <i class="fas fa-user mr-2"></i>Customer Information
            </h4>
            <div class="space-y-2">
                <div class="flex items-center">
                    <i class="fas fa-envelope mr-2 text-gray-400"></i>
                    <span class="text-sm"><?php echo htmlspecialchars($order['customer_email']); ?></span>
                </div>
                <?php if (!empty($order['customer_name'])): ?>
                <div class="flex items-center">
                    <i class="fas fa-user mr-2 text-gray-400"></i>
                    <span class="text-sm"><?php echo htmlspecialchars($order['customer_name']); ?></span>
                </div>
                <?php endif; ?>
                <?php if (!empty($order['review_location'])): ?>
                <div class="flex items-center">
                    <i class="fas fa-map-marker-alt mr-2 text-gray-400"></i>
                    <span class="text-sm"><?php echo htmlspecialchars($order['review_location']); ?></span>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Service Details -->
        <div class="bg-gray-50 p-4 rounded-lg">
            <h4 class="text-md font-medium text-gray-900 mb-3">
                <i class="fas fa-shopping-cart mr-2"></i>Service Details
            </h4>
            <div class="space-y-2">
                <div class="flex items-center">
                    <i class="fas fa-tag mr-2 text-gray-400"></i>
                    <span class="text-sm font-medium"><?php echo htmlspecialchars($order['product_name']); ?></span>
                </div>
                <div class="flex items-center">
                    <i class="fas fa-shopping-cart mr-2 text-gray-400"></i>
                    <span class="text-sm">Quantity: <?php echo $order['quantity'] ?? 1; ?></span>
                </div>
                <div class="flex items-center">
                    <i class="fas fa-dollar-sign mr-2 text-gray-400"></i>
                    <span class="text-sm font-medium text-green-600">$<?php echo number_format($order['total'], 2); ?></span>
                </div>
                <?php if (!empty($order['reviews_per_day'])): ?>
                <div class="flex items-center">
                    <i class="fas fa-star mr-2 text-gray-400"></i>
                    <span class="text-sm"><?php echo $order['reviews_per_day']; ?> reviews per day</span>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Additional Information -->
    <?php if (!empty($order['customer_notes'])): ?>
    <div class="bg-blue-50 p-4 rounded-lg">
        <h4 class="text-md font-medium text-gray-900 mb-2">
            <i class="fas fa-comment mr-2"></i>Customer Notes
        </h4>
        <p class="text-sm text-gray-700"><?php echo nl2br(htmlspecialchars($order['customer_notes'])); ?></p>
    </div>
    <?php endif; ?>

    <?php if (!empty($order['promo_code'])): ?>
    <div class="bg-green-50 p-4 rounded-lg">
        <h4 class="text-md font-medium text-gray-900 mb-2">
            <i class="fas fa-tag mr-2"></i>Promo Code Used
        </h4>
        <p class="text-sm text-gray-700 font-mono"><?php echo htmlspecialchars($order['promo_code']); ?></p>
    </div>
    <?php endif; ?>

    <?php if (!empty($order['file_name'])): ?>
    <div class="bg-purple-50 p-4 rounded-lg">
        <h4 class="text-md font-medium text-gray-900 mb-2">
            <i class="fas fa-file mr-2"></i>File Information
        </h4>
        <div class="space-y-1">
            <p class="text-sm text-gray-700">File: <?php echo htmlspecialchars($order['file_name']); ?></p>
            <?php if (!empty($order['file_link'])): ?>
            <a href="<?php echo htmlspecialchars($order['file_link']); ?>" target="_blank" class="text-sm text-blue-600 hover:text-blue-800">
                <i class="fas fa-link mr-1"></i>View File
            </a>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>
</div>
